﻿/* Skript zur Demonstration der MODEL-Klausel
 * Als Benutzer SH ausführen
 */
 
/* Herleitung: Wozu benötigen wir die MODEL-Klausel? */
-- Aufbereitung der Daten für einen forecast
select p.prod_id, t.calendar_year, 
       sum(amount_sold) amount
  from sales s,
       times t,
       products p
 where s.time_id = t.time_id
   and s.prod_id = p.prod_id
   and t.calendar_year in (2000, 2001)
   and p.prod_category_desc in ('Electronics')
 group by p.prod_id, p.prod_desc, t.calendar_year
 order by calendar_year, prod_id;
 
-- Zweite Stufe: Implementierung der Geschäftsregeln
select prod_id, amount_2000, amount_2001
  from (select p.prod_id, p.prod_desc, 
               t.calendar_year,  amount_sold
          from sales s,
               times t,
               products p
         where s.time_id = t.time_id
           and s.prod_id = p.prod_id
           and t.calendar_year in (2000, 2001)
           and p.prod_category_desc in ('Electronics'))
 pivot (sum(amount_sold)
   for calendar_year in (
         2000 as amount_2000, 
         2001 as amount_2001))
 order by prod_id;
 
-- Berechnung der Verkäufe 2002
select prod_id, 2002 calendar_year,
       case
         when prod_id in (146, 139, 143)
         then amount_2000 + amount_2001
         when prod_id in (140, 147, 145)
         then (amount_2000 + amount_2001)/2
         else amount_2001 * 1.05 end amount
  from (select p.prod_id, p.prod_desc, 
               t.calendar_year,  amount_sold
          from sales s,
               times t,
               products p
         where s.time_id = t.time_id
           and s.prod_id = p.prod_id
           and t.calendar_year in (2000, 2001)
           and p.prod_category_desc in ('Electronics'))
 pivot (sum(amount_sold)
   for calendar_year in (
         2000 as amount_2000, 
         2001 as amount_2001))
 order by prod_id;
 
-- Das gleiche mit der model-Klausel
select p prod_id, y caledar_year, a amount
  from (select p.prod_id, 
               t.calendar_year, sum(amount_sold) amount
          from sales s,
               times t,
               products p
         where s.time_id = t.time_id
           and s.prod_id = p.prod_id
           and t.calendar_year in (2000, 2001)
           and p.prod_category_desc in ('Electronics')
         group by p.prod_id, t.calendar_year) 
 model return updated rows
       dimension by (calendar_year y, prod_id p)
       measures (amount a)
       rules upsert all(
         a[2002, any] = a[2001, cv()] * 1.05,
         a[2002, p in (146, 139, 143)] = 
            a[2000, cv()] + a[2001, cv()],
         a[2002, p in (140, 147, 145)] = 
            (a[2000, cv()] + a[2001, cv()])/2            
       )
 order by p;
 
/* View für die Erläuterung der MODEL-Klausel */
create or replace view sales_data as
select c.country_region_id region_id,
       p.prod_id, 
       t.calendar_year year, 
       sum(amount_sold) sum_amount,
       sum(quantity_sold) sum_quantity
  from sales s,
       times t,
       products p,
       customers cu,
       countries c
 where s.time_id = t.time_id
   and s.prod_id = p.prod_id
   and s.cust_id = cu.cust_id
   and cu.country_id = c.country_id
   and t.calendar_year in (2000, 2001)
   and p.prod_category_desc in ('Electronics')
 group by p.prod_id, t.calendar_year, c.country_region_id;
 
-- Eine erste Auswertung auf sales_data:
select r region_id, y year, a amount
  from sales_data
 where prod_id = 146
 model unique single reference
       return updated rows
       partition by (region_id r)
       dimension by (year y, prod_id p) 
       measures (sum_amount a, sum_quantity q)
       rules upsert (
         a[2002, 146] = 15000
       );
       
-- Diese Auswertung funktioniert mit symbolischer Notation nicht:
select r region_id, y year, a amount
  from sales_data
 where prod_id = 146
 model unique single reference
       return updated rows
       partition by (region_id r)
       dimension by (year y, prod_id p) 
       measures (sum_amount a, sum_quantity q)
       rules upsert (
         a[y=2002, p=146] = 15000
       );
       
-- Auswertung, die explizit einen null-Wert ersetzt
select r region_id, y year, a amount
  from sales_data
 where prod_id in (140, 146)
 model unique single reference
       return updated rows
       partition by (region_id r)
       dimension by (year y, prod_id p) 
       measures (sum_amount a, sum_quantity q)
       rules upsert (
         a[2002, 146] = nvl(a[y=2001, p=140], 15000)
       );
       
-- Beispiel für eine Auswertung mit autmoatisierter Sortierung
select r region_id, y year, a amount
  from sales_data
 where prod_id in (140, 146)
   and region_id = 52805
 model unique single reference
       return updated rows
       partition by (region_id r)
       dimension by (year y, prod_id p) 
       measures (sum_amount a, sum_quantity q)
       rules upsert automatic order (
         a[2003, 146] = a[2002, 146] * 1.05,
         a[2002, 146] = (a[2001, cv()] + a[2000, cv()]) / 2
       );
       
-- Beispiel für eine Regel ohne sortierte Zellreferenz
select p prod_id, y year, a amount
  from sales_data
 where prod_id in (140, 146)
   and region_id = 52802
 model unique single reference
       return updated rows
       partition by (region_id r)
       dimension by (year y, prod_id p) 
       measures (sum_amount a, sum_quantity q)
       rules update automatic order(
         a[any, any] = a[cv() - 1, cv()]
       )
 order by p, y;
 
-- Beispiel für eine Regel nit sortierter Zellreferenz
select p prod_id, y year, a amount
  from sales_data
 where prod_id in (140, 146)
   and region_id = 52802
 model unique single reference
       return updated rows
       partition by (region_id r)
       dimension by (year y, prod_id p) 
       measures (sum_amount a, sum_quantity q)
       rules update automatic order(
         a[any, any] order by y desc = a[cv() - 1, cv()]
       )
 order by p, y;
 
/* ITERATIONEN in der model-Klausel */
-- Bnerechnung einer Fakultät
select n, f
  from (select 1 d, 0 n, 1 f
          from dual)
 model dimension by (d)
       measures(f, n)
 rules update
       iterate(84) until n[1] = iteration_number
       (f[1] = f[1] * greatest(iteration_number, 1));
  
-- Berechnung mehrerer Fakultäten
select n, f
  from (select rownum d, rownum n, 1 f
          from all_objects
         where rownum < 17)
 model dimension by (d)
       measures(f, n)
 rules update
       iterate(84)
       (f[d] = f[cv()] *
               case 
               when iteration_number = 0 then 1
               when iteration_number > n[cv()] then 1
               else iteration_number end);
 
/* Verwendung von Referenzen */
-- Eine View mit den aktuellen Umrechnungskursen
create or replace view currency_conversion as
-- Umwandlungsabfrage der Umrechnungskurse
select k.currency_id, k.rate
  from (select httpUriType(
                 'http://www.ecb.europa.eu/stats/eurofxref/eurofxref-daily.xml').getXml() msg
          from dual) ezb,
       xmltable(
         xmlnamespaces(
           default 'http://www.ecb.int/vocabulary/2002-08-01/eurofxref'),
         '//Cube[@currency]'
         passing ezb.msg
         columns
           currency_id char(3) path '@currency',
           rate number path 'fn:translate(@rate, ".", ",")'
       ) k
union all
select 'EUR', 1 from dual;

-- eine Tabelle zum Mappen von Ländernamen auf Währungen
create table country_2_currency(
  country_id number,
  currency_id char(3 char));
  
insert into country_2_currency
select 52790, 'USD' from dual union all
select 52776, 'EUR' from dual union all
select 52789, 'GBP' from dual union all
select 52779, 'EUR' from dual;

commit;

alter table country_2_currency add constraint c2c_country_fk
foreign key (country_id) references countries(country_id);

-- Die select-Abfrage der Referenz
select co.country_id, co.country_name, cc.currency_id, cc.rate
  from currency_conversion cc
  join country_2_currency c2c on cc.currency_id = c2c.currency_id
  join countries co on c2c.country_id = co.country_id;
  
-- Select-Abfrage der Rohdaten
select t.calendar_year, co.country_id,
       sum(amount_sold) sum_amount,
       sum(quantity_sold) sum_quantity
  from sales s,
       times t,
       customers c,
       countries co
 where s.time_id = t.time_id
   and s.cust_id = c.cust_id
   and c.country_id = co.country_id
   and t.calendar_year in (2000, 2001)
   and co.country_id in (52790, 52789, 52779, 52776)
 group by t.calendar_year, co.country_id
 order by t.calendar_year;

-- Die Abfrage mit Beteiligung der Referenz
select y year, c country_id, a amount
  from (select t.calendar_year, co.country_id,
               sum(amount_sold) sum_amount,
               sum(quantity_sold) sum_quantity
          from sales s,
               times t,
               customers c,
               countries co
         where s.time_id = t.time_id
           and s.cust_id = c.cust_id
           and c.country_id = co.country_id
           and t.calendar_year in (2000, 2001)
           and co.country_id in (52790, 52789, 52779, 52776)
         group by t.calendar_year, co.country_id
         order by t.calendar_year)
 model reference c2c on
       (select co.country_id, co.country_name, cc.currency_id, cc.rate
          from currency_conversion cc
          join country_2_currency c2c 
            on cc.currency_id = c2c.currency_id
          join countries co 
            on c2c.country_id = co.country_id)
       dimension by (country_id co)
       measures (rate)
  main sales_by_country
       dimension by (calendar_year y, country_id c) 
       measures (sum_amount a, sum_quantity q)
       rules upsert (
         a[any, any] = a[cv(), cv()] * rate[cv(c)]
       );
       
/* Anwendungsbeispiel: Ersatz einer komplexen case-Anweisung durch model */
select empno, ename, job, mgr, hiredate, sal, comm, deptno
  from emp
 model dimension by (deptno, job, empno)
       measures (sal, comm, ename, mgr, hiredate)
       rules update sequential order(
         sal[any, 'CLERK', any] = sal[cv(), cv(), cv()] * 1.05,
         sal[20, 'MANAGER', any] = sal[cv(), cv(), cv()] + 125,
         comm[30, 'SALESMAN', any] = 
           greatest(comm[cv(), cv(), cv()] + 
                     sal[cv(), cv(), cv()] * 0.05, 100),
         sal[any, for job in ('ANALYST', 'PRESIDENT'), any] =
           sal[cv(), cv(), cv()] * 1.07
       );
